
///////////////////////////////////////////////////////////////////////////////
//
//   ROCKWELL SEMICONDUCTOR SYSTEMS - COMMERCIAL GPS BUSINESS
//
///////////////////////////////////////////////////////////////////////////////
//
//
//   MSG1012.C - Message 1012 Processing
// 
//
//   DESCRIPTION
//
//   Functions to decode, build, and display Zodiac 1012 messages.
//
//
//   REVISION HISTORY
//
//   $Log:   V:\Projects\Labmon\Source\Archives\MSG1012.C_V  $
//   
//      Rev 1.16   10 Nov 1998 13:57:36   phungh
//   Lbmon511: Add cF7 key combination to
//   the menu to allow choosing antenna type
//   ( passive / active ).  Msg1218 module was
//   added.
//   
//      Rev 1.15   09 Nov 1998 10:47:48   phungh
//   labmon51: Fix file logging and missing
//   logged data bugs.  Changes made to the 
//   "Pause" variable ( 1 -> 0 ) to allow main
//   loop continue.  Move the "write to file"
//   code portion out of the interrupt handler
//   so that data is not missed because of time
//   spent too long in the interrupt handler

//      Rev 1.4   Jul 09 1997 10:39:32   COLEJ
//    
//   
//      Rev 1.3   Feb 12 1997 16:05:20   COLEJ
//    
//   
//      Rev 1.2   Nov 14 1996 11:25:28   COLEJ
//    
//   
//      Rev 1.1   Aug 14 1996 09:40:30   COLEJ
//   Repositioned sequence number up one.
//   
//   
//      Rev 1.0   13 May 1996 14:52:48   GPSADMIN
//   Initial release to version control.
//
//
////////////////////////////////////////////////////////////////////////////////

#include <stdio.h>
#include <graph.h>
#include <string.h>
#include <math.h>

#include "gentypes.h"
#include "gencons.h"
#include "ztypes.h"
#include "zcons.h"   
#include "util.h" 
#include "labmon.h"
#include "display.h"

void show_filter(short);  

////////////////////////////////////////////////////////////////////////////////
//   
//   Decode a Zodiac 1012 message from the ZMsgBuf.
//
void Dec1012(tMSGBUF *ZMsgBuf, tMSG1012 *Msg)
{  
   tSHORT i;

   GetBit(&ConfigData.PowerMgmtEnable       , ZMsgBuf,  9, 0); // 1=enabled
   GetBit(&ConfigData.ColdStartDisable      , ZMsgBuf,  9, 1); // 1=disabled
   GetBit(&ConfigData.DGPSDisabled          , ZMsgBuf,  9, 2); // 1=disabled
   GetBit(&ConfigData.HeldAltitudeDisabled  , ZMsgBuf,  9, 3); // 1=disabled
   GetBit(&ConfigData.GTSmoothingDisabled   , ZMsgBuf,  9, 4); // 1=disabled
   GetBit(&ConfigData.PosPinningDisabled    , ZMsgBuf,  9, 5); // 1=disabled
   GetBit(&ConfigData.LowQualityMeasDisabled, ZMsgBuf,  9, 6); // 1=disabled
   GetBit(&ConfigData.JammingDetectEnabled  , ZMsgBuf,  9, 7); // 1=enabled
   GetBit(&ConfigData.ActiveAntenna         , ZMsgBuf,  9, 8); // 1=present
   
   ConfigData.CNThreshold = (Msg->Configuration >> 9) & 0x00EF;
   
   for(i=0; i<16; i++){
   GetBit(&CandidatePRN[i]             , ZMsgBuf, 13, i); // ND
   GetBit(&CandidatePRN[i+16]          , ZMsgBuf, 14, i); // 1=candidate  
   }
   
   GetBit(&SolValidData.AltNotUsedReq  , ZMsgBuf, 15, 0); // 1=altitude used
   GetBit(&SolValidData.DGPSRequired   , ZMsgBuf, 15, 1); // 1=no DGPS
   GetBit(&SolValidData.DRMeasRequired , ZMsgBuf, 15, 2); // 1=no DR Measurements
   GetBit(&SolValidData.ConGPSRequired , ZMsgBuf, 15, 3); // 1=no Concurrent DR/GPS
   GetBit(&SolValidData.GPSOnlyRequired, ZMsgBuf, 15, 4); // 1=no GPS
}
                                     
////////////////////////////////////////////////////////////////////////////////
//   
//   Build a Zodiac 1012 message in the ZMsgBuf. 
//
void Bld1012(tMSGBUF *ZMsgBuf, tMSG1012 *Msg)
{
   // ack flags assumed preset

   PutShort  (0x81FF                       , ZMsgBuf, 1);      // message preamble
   PutShort  (1012                         , ZMsgBuf, 2);      // message number
   PutShort  (16                           , ZMsgBuf, 3);      // message length 
 //PutShort  (CheckSum(ZMsgBuf[0], 4)      , ZMsgBuf, 5);      // header check sum 
   
 //PutShort  (CheckSum(ZMsgBuf, 6, 48)      , ZMsgBuf, 54);    // data check sum 
}

////////////////////////////////////////////////////////////////////////////////
//   
//   Display a Zodiac 1012 message from the ZMsgBuf.
//
void Show1012(tMSG1012 *Msg)
{
   char    Buf[80] = "";
//   char    *PlatformType[7] = 
//                      {" DEF", " STA", " PED", "LAKE", " SEA", "AUTO", " AIR"};
   char    *PlatformType[7] = 
                      {" DEF", " STA", " LOW", "LAKE", " SEA", " MED", "HIGH"};
   char    *AntennaType[2] = {"PAS", "ACT"};
   char    Enabled[9], SolutionValidity[9], PlatformName[5], AntennaName[5];
   char    DisabledSV[3];
   float   ElevationMask = 0;
   tSHORT  i;
   
   extern short  xtract;  
   extern char   xtractid[4]; 
   
   extern FILE   *stream3;
   
   // convert values
   ElevationMask = (float) (Msg->ElevationMask / 1E+03 * R2D);
   
   // perform special processing for output 
   
   // navigation validity
   sprintf(SolutionValidity, "%2d      ", Msg->SatsInTrackReq); 
   if(SolValidData.AltNotUsedReq  ) SolutionValidity[2] = 'A';
   if(SolValidData.DGPSRequired   ) SolutionValidity[3] = 'D';
   if(SolValidData.DRMeasRequired ) SolutionValidity[4] = 'M';
   if(SolValidData.ConGPSRequired ) SolutionValidity[5] = 'C';
   if(SolValidData.GPSOnlyRequired) SolutionValidity[6] = 'G';


   // navigation solution type and mode
   strcpy(Enabled, "        ");
   if( ConfigData.PowerMgmtEnable       ) Enabled[0] = 'W';
   if(!ConfigData.ColdStartDisable      ) Enabled[1] = 'C';
   if(!ConfigData.DGPSDisabled          ) Enabled[2] = 'D';
   if(!ConfigData.HeldAltitudeDisabled  ) Enabled[3] = 'A';
   if(!ConfigData.GTSmoothingDisabled   ) Enabled[4] = 'S';
   if(!ConfigData.PosPinningDisabled    ) Enabled[5] = 'P';
   if(!ConfigData.LowQualityMeasDisabled) Enabled[6] = 'L';
   if( ConfigData.JammingDetectEnabled  ) Enabled[7] = 'J';
 
   // platform and antenna type
   sprintf(PlatformName,"%04s", PlatformType[Msg->ApplicationPlatform]);
   sprintf(AntennaName ,"%03s", AntennaType [ConfigData.ActiveAntenna]);

   // write extracted data to a file
   if(xtract){
      if(strcmp(xtractid,"SET") == 0){     
         //fprintf(stream3, "%6ld \t%10.6f \t%11.6f \t%9.2f\n",      
         //        Msg->GPSTimeSeconds, Longitude, Latitude, Height);                     
      }
   }                                

   // output the data items   
   sprintf(Buf,"%05d"  ,Msg->SequenceNumber         );ShowText(Buf,TIMR+ 6,TIMC+ 4);
   sprintf(Buf,"%7s"   ,SolutionValidity            );ShowText(Buf,SETR   ,SETC+ 5);
   sprintf(Buf,"%8s"   ,Enabled                     );ShowText(Buf,SETR+ 1,SETC+ 5);
   sprintf(Buf,"%5s"   ,PlatformName                );ShowText(Buf,SETR+ 2,SETC+ 6);
   sprintf(Buf,"%4s"   ,AntennaName                 );ShowText(Buf,SETR+ 2,SETC+ 9);
   sprintf(Buf,"%5d"   ,Msg->ColdStartTimeOut       );ShowText(Buf,SETR+ 4,SETC+ 6);
   sprintf(Buf,"%5d"   ,Msg->DGPSCorrTimeOut        );ShowText(Buf,SETR+ 5,SETC+ 6);
   sprintf(Buf,"%5.0f" ,Msg->MaxHorizPosError /1E+02);ShowText(Buf,SETR+ 6,SETC+ 6);
   sprintf(Buf,"%5.0f" ,Msg->MaxVertPosError  /1E+02);ShowText(Buf,SETR+ 7,SETC+ 6);
   sprintf(Buf,"%4.1f" ,ElevationMask               );ShowText(Buf,DOPR   ,DOPC+ 6);
   sprintf(Buf,"%2d"   ,ConfigData.CNThreshold		);ShowText(Buf,DOPR+ 6,DOPC+ 6);

   // update display
   _settextposition(VISR+12,VISC+4);
   _outtext("           ");
   
   Buf[0] = '\0';
   for(i=0; i<32; i++){    
      if(CandidatePRN[i] == 0){     
          sprintf(DisabledSV, "%2d ", i+1);
          strcat(Buf, DisabledSV);
      }
   } 
   
   _settextposition(VISR+12,VISC+4);
   _outtext(Buf);  
}                                                 
                                      
                                           
